﻿/*  Author:     Ben Hall
 *  File:       AttackerAI.cs
 *  Date:       4/28/2011
 *  
 *  Purpose:    AttackerAI.cs extends the AICharacter class and overrides the abstract method MakeMove()
 *              using a decision tree that focuses on making the strongest offensive moves available
 *              to the AICharacter.
 *              
 *              The MakeMove() method also uses a random chance to skip the "best" move the AICharacter
 *              can make at each branch, based on the AICharacter's intelligence, with the chance of it
 *              skipping a "best" move decreasing every time the "best" move is skipped.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace OnlineGame
{
    public class AttackerAI : AICharacter
    {
        public AttackerAI(String name, int strIn, int stmIn, int aglIn, int intIn, int fthIn, int prsIn, int currencyIn)
        {
            this.characterName = name;
            this.strength = new CharStat(strIn);
            this.stamina = new CharStat(stmIn);
            this.agility = new CharStat(aglIn);
            this.intelligence = new CharStat(intIn);
            this.faith = new CharStat(fthIn);
            this.persona = new CharStat(prsIn);

            this.currencyCarried = currencyIn;

            this.maxHealth = DetermineMaxHealth();
            this.currentHealth = maxHealth;
            this.maxMagic = DetermineMaxMagic();
            this.currentMagic = maxMagic;

            this.defending = false;
            this.poisoned = false;
            this.covering = false;
            this.coveredBy = null;

            this.skillsKnown = new bool[MainPage.skillList.Length];
            for (int i = 0; i < skillsKnown.Length; i++)
            {
                skillsKnown[i] = false;
            }

            this.decisionChance = DetermineDecisionChance();
        }

        /*  MakeMove() selects an action for the AICharacter to take, using a decision tree with a
         *  random chance of skipping the "best move" at each branch (based on the AICharacter's intelligence
         *  score)
         *  
         *  AttackerAI prioritizes doing direct damage to its opponent and may do supportive actions
         *  to itself, but does not concern itself with the state of its allies.
         */
        public override String MakeMove()
        {
            String messageReturned = "";

            //variables needed to determine if an AI skips a more optimal move
            //initialize decisionChance to the base value, based on intelligence stat
            this.decisionChance = DetermineDecisionChance();
            this.rndDecision = MainPage.rnd.NextDouble();
            //AI may decide to skip more optimal moves, with lower intelligence being more likely to skip moves
            bool useBestMove = false;
            useBestMove = (rndDecision < decisionChance);

            int chosenSkill = GetCompHealSkill();  // will return -1 if no relevant skill is available
            if (((this.GetHealthPercent() < lowHealth) && (chosenSkill >= 0)) && (useBestMove))
            {
                //AI uses a healing skill if available, when health is under 25%
                messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
            }
            else
            {
                if (!useBestMove)
                {
                    useBestMove = RedetermineUseBestMove();
                }

                chosenSkill = GetCompAttackSkill();  //will return -1 if no relevant skill is available
                if ((chosenSkill >= 0) && (useBestMove))
                {
                    //AI uses a damaging skill if available if health is above 25%
                    messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, MainPage.playerCharacter);
                }
                else
                {
                    if (!useBestMove)
                    {
                        useBestMove = RedetermineUseBestMove();
                    }

                    if (this.GetHealthPercent() >= halfHealth)
                    {
                        //AI uses Fight if health is above 50% and AI does not use a damaging skill
                        messageReturned = MainPage.FightCommand(this, MainPage.playerCharacter);
                    }
                    else
                    {
                        if (this.IsPoisoned() && (useBestMove))
                        {
                            if (!useBestMove)
                            {
                                useBestMove = RedetermineUseBestMove();
                            }

                            chosenSkill = GetCompRemovePoisonSkill();
                            if ((chosenSkill >= 0) && (useBestMove))
                            {
                                //AI heals its poison if its health is between 25% and 50% and it does not attack
                                messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
                            }
                            else
                            {
                                //40% chance of healing self (if possible), 60% of Fight
                                double random = MainPage.rnd.NextDouble();
                                chosenSkill = GetCompHealSkill();
                                if ((random < 0.4) && (chosenSkill >= 0))
                                {
                                    messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
                                }
                                else
                                {
                                    messageReturned = MainPage.FightCommand(this, MainPage.playerCharacter);
                                }
                            }
                        }
                        else //when not poisoned
                        {
                            //40% chance of Defensive action, 60% chance of Fight
                            double random = MainPage.rnd.NextDouble();
                            if (random < 0.4)
                            {
                                //defend
                                messageReturned = UseBestDefense();
                            }
                            else
                            {
                                messageReturned = MainPage.FightCommand(this, MainPage.playerCharacter);
                            }
                        }
                    }
                }
            }
            return messageReturned;
        }

    }
}
